#ifndef _IIC_H
#define _IIC_H

/*includes*/
#include "iic_config.h"


/*defines*/

#define MWSR                   0x00  /* Master write  */
#define MRSW                   0x01  /* Master read */

#define IIC_INTERRUPT_ENABLE  1    //enables the interrupt for IIC
#define IIC_INTERRUPT_DISABLE 0    //disables the interrupt for IIC

#define IIC_10_BIT_SLAVE_ADDR 1    //10 bits used for slave address
#define IIC_7_BIT_SLAVE_ADDR  0    //7 bits used for slave address

#define IIC_MWSR                   0x00  /* Master write slave read */
#define IIC_MRSW                   0x01  /* Master read slave write*/


/******************************************************************************/
/* IIC_Enter_TxMode                                                           */
/******************************************************************************/
/* Abstract          : This function configures master in transmit mode       */
/*					                                                                  */
/*                                                                            */
/* Input Parameters  : None                                                   */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/

#define IIC_Enter_RxMode()    IICC1_TX = 0

/******************************************************************************/
/* IIC_Enter_RxMode                                                           */
/******************************************************************************/
/* Abstract          : This function configures master in receive mode        */
/*					                                                                  */
/*                                                                            */
/* Input Parameters  : None                                                   */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/

#define IIC_Enter_TxMode()    IICC1_TX = 1

/******************************************************************************/
/* FUNCTION          : IIC_Send_Start                                         */
/******************************************************************************/
/* Abstract          : This function sends the start symbol for starting      */
/*					           Tx/Rx transaction                                      */
/*                                                                            */
/* Input Parameters  : None                                                   */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

#define IIC_Send_Start()      IICC1_TX = 1;\
                              IICC1_MST = 1   //when it changes from slave mode to master mode start signal is transmitted

/******************************************************************************/
/* FUNCTION          : IIC_Repeated_Start                                     */
/******************************************************************************/
/* Abstract          : This function sends repeated start symbol on SDA for   */
/*					           continuing data transfer without bus rentention loss   */
/*                                                                            */
/* Input Parameters  : None                                                   */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

#define IIC_Repeated_Start()  IICC1_RSTA= 1   //Writing 1 to this bit generates a repeated START provided it is the current master.
                                              //Attempting a repeat at the wrong time results in loss of arbitration.

/******************************************************************************/
/* FUNCTION          : IIC_Send_Stop                                          */
/******************************************************************************/
/* Abstract          : This function sends the STOP signal over SDA to stop   */
/*					           Tx/Rx transaction and release IIC bus                  */
/*                                                                            */
/* Input Parameters  : None                                                   */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

#define IIC_Send_Stop()       IICC1_MST = 0  //when it changes from master mode to slave mode it sends stop signal

/******************************************************************************/
/* FUNCTION          : IIC_Enable_Ack                                      */
/******************************************************************************/
/* Abstract          : This function enables the send acknowledge procedure   */
/*					           by configuring the internal registers                  */
/*                                                                            */
/* Input Parameters  : None                                                   */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

#define IIC_Enable_Ack()      IICC1 |= TX_ACK_EN

/******************************************************************************/
/* FUNCTION          :IIC_Disable_Ack                                      */
/******************************************************************************/
/* Abstract          : This function disables the send acknowledge procedure  */
/*					           by configuring the internal registers                  */
/*                                                                            */
/* Input Parameters  : None                                                   */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

#define IIC_Disable_Ack()     IICC1 |= TX_ACK_DISABLE 

/******************************************************************************/
/* FUNCTION          : IIC_Write_Byte                                      */
/******************************************************************************/
/* Abstract          : This function is used for master write transaction     */
/*					                                                                  */
/*                                                                            */
/* Input Parameters  : None                                                   */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

#define IIC_Write_Byte(data)  IICD = data

/*functions prototypes*/

/******************************************************************************/
/* FUNCTION          : IIC_Init                                               */
/******************************************************************************/
/* Abstract          : This function initializes i2c interface                */
/*					           by configuring the internal registers                  */
/*                                                                            */
/* Input Parameters  : Interrupt_Enable: IIC_INTERRUPT_ENABLE                 */
/*                                       /IIC_INTERRUPT_DISABLE               */
/*                     Addr_No_Of_Bits : IIC_10_BIT_SLAVE_ADDR                */ 
/*                                       /IIC_7_BIT_SLAVE_ADDR                */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/

void IIC_Init(unsigned char Interrupt_Enable,unsigned char Addr_No_Of_Bits,void (*p)(void));

/******************************************************************************/
/* FUNCTION          : IIC_StartTransmission                                  */
/******************************************************************************/
/* Abstract          : This function starts the transaction of read/write     */
/*					           with a slave                                           */
/*                                                                            */
/* Input Parameters  : SlaveAddress: Address of destination slave             */
/*                     TxRxMode: Specifies the mode Tx or Rx                  */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

void IIC_Start_Transmission(unsigned int SlaveAddress,unsigned char TxRxMode);

/******************************************************************************/
/* FUNCTION          : IIC_Wait_Ack                                      */
/******************************************************************************/
/* Abstract          : This function is called when waiting for some data/ack */
/*					           It waits on IIC flag and clears it afterwards          */
/*                                                                            */
/* Input Parameters  : None                                                   */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/

void IIC_Wait_Ack(void);


#endif /*_IIC_FLOWTRON_H ends*/ 